// TBBSum.cpp 
//
#include <algorithm>
#include <limits>
#include <vector>
#include <iostream>
#include <functional>

#include "tbb/blocked_range.h"
#include "tbb/task_scheduler_init.h"
#include "tbb/parallel_reduce.h"
#include "tbb/parallel_for.h"
#include "tbb/tick_count.h"

// on windows there is a macro "min" in windows.h defined
#undef min

struct SerialScale {
	double factor;
	SerialScale(const double &f):factor(f){}
	void operator()(double &d) 
	{ d *= factor; }
};

struct ParallelScale
{
	const double& factor;

	ParallelScale(const double& f ) : factor( f ) {}

	typedef tbb::blocked_range< std::vector<double>::iterator > range_type;

	void operator()( const range_type& range ) const
	{
		for ( std::vector<double>::iterator it=range.begin(); it!=range.end(); ++it )
			*it *= factor;
	}
};

struct ParallelMax
{
	double myMax;

	ParallelMax() :
		myMax( std::numeric_limits<double>::min() )
	{}

	ParallelMax( ParallelMax& m, tbb::split ) :
		myMax( std::numeric_limits<double>::min() )
	{}

	void operator()( const tbb::blocked_range< std::vector<double>::iterator >& range ) {
		for ( std::vector<double>::iterator it=range.begin(); it!=range.end(); ++it )
			if ( *it > myMax )
			{ myMax = *it; }
	}

	void join( const ParallelMax& rhs ) {
		if ( rhs.myMax > myMax )
		{ myMax = rhs.myMax; }
	}
};

int main(int argc, char* argv[])
{
	std::vector<double> theVector;

	tbb::task_scheduler_init init;

	theVector.resize( 10000 );
	std::generate( theVector.begin(), theVector.end(), rand );

	tbb::blocked_range< std::vector<double>::iterator > theRange( theVector.begin(), theVector.end(), 1000 );

	ParallelMax someMax0;
	tbb::parallel_reduce( theRange, someMax0 );
	std::cout << theVector[0] << " Maximum = " << someMax0.myMax << std::endl;
	
	SerialScale theScaling(2.0);
	std::for_each( theVector.begin(), theVector.end(), theScaling);
	ParallelMax someMax1;
	tbb::parallel_reduce( theRange, someMax1 );
	std::cout << theVector[0] << " Maximum = " << someMax1.myMax << std::endl;

	ParallelScale someScale(5.0);
	tbb::parallel_for( theRange, someScale );
	ParallelMax someMax2;
	tbb::parallel_reduce( theRange, someMax2 );
	std::cout << theVector[0] << " Maximum = " << someMax2.myMax << std::endl;

	return 0;
}
